<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Post;
use App\Models\Category;

class AdminController extends Controller
{
    // 1. Dashboard Page
    public function dashboard()
    {
        // Thoda stats dikha denge dashboard par
        $totalPosts = Post::count();
        $jobs = Post::where('post_type', 'job')->count();
        $admitCards = Post::where('post_type', 'admit_card')->count();
        
        return view('admin.dashboard', compact('totalPosts', 'jobs', 'admitCards'));
    }

    // 2. All Posts List
    public function posts(Request $request)
    {
        $query = Post::query();

        // Search Logic
        if ($request->has('search')) {
            $query->where('title', 'like', '%' . $request->search . '%');
        }

        // Pagination ke saath data lao (20 per page)
        $posts = $query->latest()->paginate(20);

        return view('admin.posts.index', compact('posts'));
    }
    // --- 3. SHOW CREATE POST FORM ---
    public function createPost()
    {
        // Category dropdown ke liye chahiye
        $categories = Category::where('status', 1)->get(); 
        return view('admin.posts.create', compact('categories'));
    }

    // --- 4. SAVE POST TO DATABASE ---
   public function storePost(Request $request)
    {
        $request->validate([
            'title' => 'required|max:255',
            'slug' => 'required|unique:posts,slug',
            'post_type' => 'required',
            'category_id' => 'required',
        ]);

        $post = new Post();
        $post->title = $request->title;
        $post->slug = Str::slug($request->slug);
        $post->post_type = $request->post_type;
        $post->category_id = $request->category_id;
        $post->short_desc = $request->short_desc;
        
        // Keyword Logic: Agar khali hai toh Title hi Keyword banega
        $post->post_name = $request->post_name ?? $request->title;

        // Content Areas
        $post->dates_content = $request->dates_content;
        $post->fees_content = $request->fees_content;
        $post->age_content = $request->age_content;
        $post->vacancy_content = $request->vacancy_content;
        $post->links_content = $request->links_content;
        
        // Links
        $post->official_website = $request->official_website;
        $post->telegram_link = $request->telegram_link;

        // Checkboxes
        $post->is_featured = $request->has('is_featured') ? 1 : 0;
        $post->is_marquee = $request->has('is_marquee') ? 1 : 0;

        // --- 🔥 AUTO SEO LOGIC (MOVED BEFORE SAVE) ---
        
        // 1. Meta Title Generation
        if (!$request->meta_title) {
            $siteName = get_setting('site_title', 'Sarkari Result');
            $post->meta_title = $request->title . ' - Apply Online, Notification @ ' . $siteName;
        } else {
            $post->meta_title = $request->meta_title;
        }

        // 2. Meta Description Generation
        if (!$request->meta_desc) {
            $post->meta_desc = "Apply Online for " . $request->title . ". Check Eligibility, Exam Date, Age Limit, Syllabus, and Official Notification details at " . get_setting('site_title');
        } else {
            $post->meta_desc = $request->meta_desc;
        }
        // ---------------------------------------------

        $post->save(); // Ab Data + SEO save hoga!

        return redirect()->route('admin.posts.index')->with('success', 'Post created successfully!');
    }
    // --- 5. EDIT FORM SHOW KARNA ---
    public function editPost($id)
    {
        $post = Post::findOrFail($id);
        $categories = Category::where('status', 1)->get();
        return view('admin.posts.edit', compact('post', 'categories'));
    }

   // --- 6. UPDATE POST (FIXED LOGIC) ---
    public function updatePost(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|max:255',
            'slug' => 'required|unique:posts,slug,' . $id,
            'post_type' => 'required',
            'category_id' => 'required',
        ]);

        $post = Post::findOrFail($id);

        $post->title = $request->title;
        $post->slug = Str::slug($request->slug);
        $post->post_name = $request->post_name;
        $post->short_desc = $request->short_desc;
        $post->post_type = $request->post_type;
        $post->category_id = $request->category_id;
        
        $post->is_featured = $request->has('is_featured') ? 1 : 0;
        $post->is_marquee = $request->has('is_marquee') ? 1 : 0;
        
        $post->dates_content = $request->dates_content;
        $post->fees_content = $request->fees_content;
        $post->age_content = $request->age_content;
        $post->vacancy_content = $request->vacancy_content;
        $post->links_content = $request->links_content;
        
        $post->official_website = $request->official_website;
        $post->telegram_link = $request->telegram_link;

        // --- 🔥 AUTO SEO LOGIC (MOVED BEFORE SAVE) ---
        
        // Update Title if user provided one, else keep old or generate new
        if ($request->meta_title) {
            $post->meta_title = $request->meta_title;
        } elseif (!$post->meta_title) {
             // Agar purana bhi khali tha, naya banao
            $post->meta_title = $request->title . ' - Info';
        }

        // Update Description
        if ($request->meta_desc) {
            $post->meta_desc = $request->meta_desc;
        } elseif (!$post->meta_desc) {
            $post->meta_desc = Str::limit(strip_tags($request->short_desc), 160);
        }
        
        $post->last_updated = now(); // Update Time
        // ---------------------------------------------

        $post->save();

        return redirect()->route('admin.posts.index')->with('success', 'Post updated successfully!');
    }
    // --- 7. DELETE POST ---
    public function deletePost($id)
    {
        $post = Post::findOrFail($id);
        $post->delete();
        return redirect()->back()->with('success', 'Post deleted successfully!');
    }
    // ==========================================
    //           CATEGORY MANAGEMENT
    // ==========================================

    // 1. Show All Categories
    public function categories()
    {
        $categories = Category::latest()->get();
        return view('admin.categories.index', compact('categories'));
    }

    // 2. Store New Category
    public function storeCategory(Request $request)
    {
        $request->validate([
            'name' => 'required|unique:categories,name',
            'slug' => 'required|unique:categories,slug',
        ]);

        $cat = new Category();
        $cat->name = $request->name;
        $cat->slug = $request->slug;
        $cat->save();

        return back()->with('success', 'Category added successfully!');
    }

    // 3. Edit Category
    public function editCategory($id)
    {
        $category = Category::findOrFail($id);
        $allCategories = Category::latest()->get(); // List bhi dikhayenge taaki edit karte waqt list gayab na ho
        return view('admin.categories.index', compact('category', 'allCategories'));
    }

    // 4. Update Category
    public function updateCategory(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|unique:categories,name,'.$id,
            'slug' => 'required|unique:categories,slug,'.$id,
        ]);

        $cat = Category::findOrFail($id);
        $cat->name = $request->name;
        $cat->slug = $request->slug;
        $cat->save();

        return redirect()->route('admin.categories.index')->with('success', 'Category updated successfully!');
    }

    // 5. Delete Category
    public function deleteCategory($id)
    {
        Category::findOrFail($id)->delete();
        return back()->with('success', 'Category deleted successfully!');
    }
    // ==========================================
    //           PAGES MANAGEMENT
    // ==========================================

    // 1. Show All Pages
    public function pages()
    {
        $pages = \App\Models\Page::latest()->get();
        return view('admin.pages.index', compact('pages'));
    }

    // 2. Show Create Form
    public function createPage()
    {
        return view('admin.pages.create');
    }

    // 3. Store New Page
    public function storePage(Request $request)
    {
        $request->validate([
            'title' => 'required|max:255',
            'slug' => 'required|unique:pages,slug',
        ]);

        $page = new \App\Models\Page();
        $page->title = $request->title;
        $page->slug = $request->slug;
        $page->content = $request->content; // Summernote content
        $page->status = $request->has('status') ? 1 : 0;
        $page->save();

        return redirect()->route('admin.pages.index')->with('success', 'Page created successfully!');
    }

    // 4. Show Edit Form
    public function editPage($id)
    {
        $page = \App\Models\Page::findOrFail($id);
        return view('admin.pages.edit', compact('page'));
    }

    // 5. Update Page
    public function updatePage(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|max:255',
            'slug' => 'required|unique:pages,slug,' . $id,
        ]);

        $page = \App\Models\Page::findOrFail($id);
        $page->title = $request->title;
        $page->slug = $request->slug;
        $page->content = $request->content;
        $page->status = $request->has('status') ? 1 : 0;
        $page->save();

        return redirect()->route('admin.pages.index')->with('success', 'Page updated successfully!');
    }

    // 6. Delete Page
    public function deletePage($id)
    {
        \App\Models\Page::findOrFail($id)->delete();
        return back()->with('success', 'Page deleted successfully!');
    }
    // ==========================================
    //           GENERAL SETTINGS
    // ==========================================

    // 1. Show Settings Page
    public function settings()
    {
        // Hum saari settings ko ek array mein convert kar lenge taaki view mein aasaani se mile
        // Format: ['site_title' => 'Sarkari Result', 'footer_text' => 'Copyright...']
        $settings = \App\Models\Setting::pluck('value', 'key')->toArray();
        
        return view('admin.settings.index', compact('settings'));
    }

    // 2. Update Settings (FULL FUNCTION)
    public function updateSettings(Request $request)
    {
        // --- 1. Text Settings Save Karo ---
        $keys = [
            'site_title', 
            'meta_description', 
            'footer_text', 
            'social_facebook', 
            'social_telegram', 
            'header_script'
        ];
        
        foreach ($keys as $key) {
            // Hum check karenge ki form mein ye key aayi hai ya nahi
            if ($request->has($key)) {
                \App\Models\Setting::updateOrCreate(
                    ['key' => $key], // Match this key
                    ['value' => $request->input($key), 'type' => 'text'] // Update this value
                );

                // 🔥 CRITICAL: Cache Clear karo taaki turant asar dikhe
                // Agar ye nahi karoge to browser purana naam dikhata rahega
                \Illuminate\Support\Facades\Cache::forget('setting_' . $key);
            }
        }

        // --- 2. Image Settings (Logo & Favicon) ---
        $imageKeys = ['site_logo', 'favicon'];
        
        foreach ($imageKeys as $key) {
            if ($request->hasFile($key)) {
                // Image Upload Logic
                $path = $request->file($key)->store('settings', 'public');
                
                \App\Models\Setting::updateOrCreate(
                    ['key' => $key],
                    ['value' => $path, 'type' => 'image']
                );

                // 🔥 CRITICAL: Image ka cache bhi clear karo
                \Illuminate\Support\Facades\Cache::forget('setting_' . $key);
            }
        }

        return back()->with('success', 'Settings updated successfully!');
    }
    
    // Create, Store, Edit functions hum agle step me likhenge...
}